#!/bin/sh

###############################################################################
# Script: Zoho Assist Bulk Deployment
# Description: Automates the download, installation, and configuration of Zoho Assist.
# Author: ZohoCorp
# Date: 19-02-2025
###############################################################################

# Deployment Link
downloadLink='Deployment Link'

# Extract encapiKey and domain from the download link
encapiKey=$(echo "$downloadLink" | awk -F'encapiKey=' '{print $2}' | awk -F'&' '{print $1}')
domain=$(echo "$downloadLink" | awk -F/ '{print $1 "//" $3}')

# Construct the download URL
downloadURL="${domain}/api/v2/install_urs?type=2&encapiKey=${encapiKey}&app=mac"

# Define mount point for the DMG
mountPoint="/Volumes/Remote Access Installer"

# Get the currently logged-in user
loggedInUser=$(stat -f "%Su" /dev/console)

# Fallback methods to get the logged-in user if the above fails
if [ -z "$loggedInUser" ]; then
    loggedInUser=$(scutil <<< "show State:/Users/ConsoleUser" | awk '/Name :/ && ! /loginwindow/ {print $3}')
fi

if [ -z "$loggedInUser" ]; then
    loggedInUser=$(echo "show State:/Users/ConsoleUser" | scutil | awk '/Name :/ && ! /loginwindow/ {print $3}')
fi

# Log the logged-in user
echo "The logged-in user is: $loggedInUser"

# Debugging: Print extracted values
echo "encapiKey: $encapiKey"
echo "Domain: $domain"
echo "Download URL: $downloadURL"

# Clean up and create working directory
WORK_DIR="/Library/ZohoAssist"
rm -rf "$WORK_DIR"
mkdir -p "$WORK_DIR"
cd "$WORK_DIR" || { echo "Failed to change directory to $WORK_DIR"; exit 1; }

# Unmount the DMG if already mounted
if mount | grep -q "$mountPoint"; then
    hdiutil unmount "$mountPoint" || { echo "Failed to unmount $mountPoint"; exit 1; }
fi

# Download the Zoho Assist Installer
echo "Downloading Zoho Assist Installer..."
curl -f -L "$downloadURL" --output ZohoAssistInstaller.dmg || { echo "Failed to download Zoho Assist Installer"; exit 1; }

# Mount the DMG
echo "Mounting Zoho Assist Installer DMG..."
hdiutil attach -nobrowse ZohoAssistInstaller.dmg || { echo "Failed to mount DMG"; exit 1; }

# Wait for the DMG to mount (strategic sleep)
sleep 2

# Run the installer
echo "Running Zoho Assist Installer..."
"/Volumes/Remote Access Installer/InstallerSetup.app/Contents/MacOS/InstallerSetup" || { echo "Installation failed"; exit 1; }

# Wait for the installer to complete (strategic sleep)
sleep 3

# Load Launch Agents
echo "Loading Launch Agents..."
sudo -u "$loggedInUser" launchctl load /Library/LaunchAgents/com.zoho.assist.PerUserAgent.plist
sudo -u "$loggedInUser" launchctl load /Library/LaunchAgents/com.zoho.assist.ZAPreLogin.plist

# Wait for Launch Agents to load (strategic sleep)
sleep 2

# Unmount the DMG
echo "Unmounting DMG..."
hdiutil unmount -force "$mountPoint" || { echo "Failed to unmount DMG"; exit 1; }

# Clean up
echo "Cleaning up..."
rm -rf "$WORK_DIR"

# Configure Accessibility Permissions
echo "Configuring Accessibility Permissions..."
cd /Library/Application\ Support/ZohoAssist/ || { echo "Failed to change directory to /Library/Application Support/ZohoAssist/"; exit 1; }

open Zoho.app --args -appMode CHECK_ACCESS
sleep 2
open Zoho.app --args -appMode CHECK_RECORD_ACCESS

echo "Zoho Assist deployment completed successfully."
exit 0
