const CommonUtil = require("../utils/commonUtil");	//NO I18N
const DatabaseUtil = require("../utils/databaseUtil");	//NO I18N

const Functions = (function () {
    let reqData;
    let reqParams;
    const _handler = async (data) => {
        _inititalize(data);
        let response = {};
        const handler = reqData.handler.type;
        switch (handler) {
            case "button_handler":	//NO I18N
                response = await _buttonHandler();
                break;
            case "form_handler":	//NO I18N
                response = await _formHandler();
                break;
        }
        return response;
    };

    const _inititalize = async (data) => {
        reqData = data;
        reqParams = data.params;
    };

    const _buttonHandler = async () => {
        if (reqData.name === "timetrackerbtn") {
            switch (reqParams.target.name) {
                case "Authorize":	//NO I18N
                    return await _handlerUserToken();
                case "Create Project":	//NO I18N
                    return _createProject();
                case "View":	//NO I18N
                    return await _viewTasks();
                default:
                    return await _handleButtonActions(reqParams.target.name);
            }
        }
    };

    const _formHandler = async () => {
        try {
            const formName = reqParams.form.name;
            const formValues = reqParams.form.values;
            const zuid = reqParams.access.user_id;
            let response = {};
            if (formName === "addtask") {
                if (CommonUtil.isEmptyString(formValues.name)) {
                    return {
                        type: "form_error",	//NO I18N
                        inputs: { name: "Task name cannot be empty." }	//NO I18N
                    };
                }

                const now = Date.now();
                const taskData = {
                    creator: zuid,
                    name: formValues.name.trim(),
                    projectId: formValues.project.value,
                    startTime: now,
                    status: "started"	//NO I18N
                };
                const taskId = await DatabaseUtil.tasks.addTask(taskData);

                let responseText = `*Task name: * ${taskData.name}\n`;
                responseText += `*Project:* ${formValues.project.label}\n\n`;
                responseText += `*Timer started at:* ${CommonUtil.getLocaleStringFromMS(
                    now,
                    "datetime"	//NO I18N
                )}`;
                CommonUtil.postMessageToBot(await DatabaseUtil.users.getToken(zuid), {
                    text: responseText,
                    card: {
                        title: ":task: Task Added - The clock started ticking",	//NO I18N
                        theme: "modern-inline"	//NO I18N
                    },
                    buttons: [
                        {
                            label: "Pause",	//NO I18N
                            action: {
                                type: "invoke.function",	//NO I18N
                                data: { name: "timetrackerbtn" }	//NO I18N
                            },
                            key: taskId
                        },
                        {
                            label: "Stop",	//NO I18N
                            type: "-",
                            action: {
                                type: "invoke.function",	//NO I18N
                                data: { name: "timetrackerbtn" }	//NO I18N
                            },
                            key: taskId
                        }
                    ]
                });
            } else if (formName === "createproject") {	//NO I18N
                let projectTitle = formValues.title;
                if (CommonUtil.isEmptyString(projectTitle)) {
                    response = {
                        type: "form_error",	//NO I18N
                        inputs: { title: "Project title cannot be empty." }	//NO I18N
                    };
                } else {
                    projectTitle = projectTitle.trim();
                    if (projectTitle.toLowerCase() === "general") {
                        response = {
                            type: "form_error",	//NO I18N
                            text: "General is the default project. Try using different title."	//NO I18N
                        };
                    } else {
                        await DatabaseUtil.projects.createProject(zuid, projectTitle);
                        response = CommonUtil.getBannerResponse("Project created.");	//NO I18N
                    }
                }
            }
            return response;
        } catch (error) {
            throw error;
        }
    };

    const _handlerUserToken = async () => {
        try {
            const zuid = reqParams.access.user_id;
            await DatabaseUtil.users.addToken(zuid, reqParams.arguments.input.token);
            return CommonUtil.getBannerResponse(
                "Authorization successful :thumbsup: "	//NO I18N
            );
        } catch (error) {
            throw error;
        }
    };

    const _createProject = () => {
        return {
            name: "createproject",	//NO I18N
            type: "form",	//NO I18N
            title: "Create Project",	//NO I18N
            button_label: "Create",	//NO I18N
            inputs: [
                {
                    type: "text",	//NO I18N
                    name: "title",	//NO I18N
                    label: "Title",	//NO I18N
                    placeholder: "Project Title",	//NO I18N
                    min_length: "3",
                    max_length: "50",
                    mandatory: true
                }
            ],
            action: { type: "invoke.function", name: "timetrackerform" }	//NO I18N
        };
    };

    const _viewTasks = async () => {
        const projectId = reqParams.arguments.key;
        const allTasks = await DatabaseUtil.tasks.getAllTasks(projectId);
        let response = {};
        if (allTasks.length > 0) {
            let taskTableRows = [];
            let taskData, taskStatus, btnActions, btnProps;
            allTasks.forEach((task) => {
                taskStatus = task.STATUS;
                taskData = {
                    Task: task.NAME,
                    "Created on": CommonUtil.getLocaleStringFromMS(	//NO I18N
                        task.CREATED_ON,
                        "date"	//NO I18N
                    ),
                    Status: CommonUtil.capitalize(taskStatus)
                };
                btnActions = "";
                btnProps = `(invoke.function|timetrackerbtn|test@test.tst|${task.TASK_ID})`;

                if (taskStatus === "started") {
                    btnActions = `[Pause]${btnProps} [Stop]${btnProps} `;
                    taskData["Time Spent"] =
                        task.TIME_SPENT === null
                            ? Date.now() - task.START_TIME
                            : task.TIME_SPENT;
                } else if (taskStatus === "paused") {	//NO I18N
                    btnActions = `[Resume]${btnProps} [Stop]${btnProps} `;
                    taskData["Time Spent"] = task.TIME_SPENT;
                } else if (taskStatus === "completed") {	//NO I18N
                    taskData["Time Spent"] = task.TIME_SPENT;
                    taskData["Completed on"] = CommonUtil.getLocaleStringFromMS(
                        task.END_TIME,
                        "datetime"	//NO I18N
                    );
                }
                btnActions += `[-Delete]${btnProps}`;
                taskData.Action = btnActions;
                taskData["Time Spent"] = CommonUtil.getDurationFromMS(
                    taskData["Time Spent"]
                );
                taskTableRows.push(taskData);
            });

            response = {
                text: "Here is the list of all tasks.",	//NO I18N
                card: { theme: "modern-inline" },	//NO I18N
                slides: [
                    {
                        type: "table",	//NO I18N
                        data: {
                            headers: [
                                "Task",	//NO I18N
                                "Created on",	//NO I18N
                                "Status",	//NO I18N
                                "Completed on",	//NO I18N
                                "Time Spent",	//NO I18N
                                "Action"	//NO I18N
                            ],
                            rows: taskTableRows
                        }
                    }
                ]
            };
        } else {
            response = CommonUtil.getBannerResponse("No tasks found.", true);	//NO I18N
        }
        return response;
    };

    const _handleButtonActions = async (action) => {
        const key = reqParams.arguments.key.split("|");
        if (key.length === 1) {
            return _handleTasks(action, key[0]);
        } else {
            return _deleteProject(key[0]);
        }
    };

    const _handleTasks = async (action, taskId) => {
        let response;
        const zuid = reqParams.access.user_id;
        const taskDetails = await DatabaseUtil.tasks.getTask(taskId);
        if (taskDetails.length === 0 || taskDetails === null) {
            if (action === "Delete") {
                response = CommonUtil.getBannerResponse(
                    "Task has already been deleted.",	//NO I18N
                    true
                );
            } else {
                response = CommonUtil.getBannerResponse("Task not found.", true);	//NO I18N
            }
        } else {
            const taskStatus = taskDetails.STATUS;
            if (action === "Resume" && taskStatus === "started") {
                response = CommonUtil.getBannerResponse(
                    "Timer for task has already been started.",	//NO I18N
                    true
                );
            } else if (action === "Pause" && taskStatus === "paused") {	//NO I18N
                response = CommonUtil.getBannerResponse(
                    "Task is already on hold.",	//NO I18N
                    true
                );
            } else if ((action === "Pause" || action === "Stop") && taskStatus === "completed") {	//NO I18N
                response = CommonUtil.getBannerResponse(
                    "Task has already been completed.",	//NO I18N
                    true
                );
            } else if (action === "Delete") {	//NO I18N
                await DatabaseUtil.tasks.deleteTask(taskId);
                response = CommonUtil.getBannerResponse("Task deleted.");	//NO I18N
            } else {
                let taskUpdate;
                let totalTimeSpent = 0;
                const now = Date.now();
                switch (action) {
                    case "Resume": {	//NO I18N
                        const currentTask = await DatabaseUtil.tasks.getCurrentTask(zuid);
                        if (currentTask !== null) {
                            return CommonUtil.getBannerResponse(
                                "Complete or pause your current task before starting a new task.",	//NO I18N
                                true
                            );
                        }
                        
                        taskUpdate = {
                            flagTime: now,
                            status: "started"	//NO I18N
                        };

                        response = CommonUtil.getBannerResponse(
                            "Task resumed."	//NO I18N
                        );
                        break;
                    }
                    case "Pause": {	//NO I18N
                        if (taskDetails.TIME_SPENT === null) {
                            totalTimeSpent = now - taskDetails.START_TIME;
                        } else {
                            totalTimeSpent =
                                parseInt(taskDetails.TIME_SPENT) +
                                parseInt(now - taskDetails.FLAG_TIME);
                        }

                        taskUpdate = {
                            flagTime: now,
                            timeSpent: totalTimeSpent,
                            status: "paused"	//NO I18N
                        };
                        response = CommonUtil.getBannerResponse("Task paused.");	//NO I18N
                        break;
                    }
                    case "Stop": {	//NO I18N
                        if (taskDetails.TIME_SPENT === null) {
                            totalTimeSpent = now - taskDetails.START_TIME;
                        } else if (taskStatus === "paused") {	//NO I18N
                            totalTimeSpent = taskDetails.TIME_SPENT;
                        } else {
                            totalTimeSpent =
                                parseInt(taskDetails.TIME_SPENT) +
                                parseInt(now - taskDetails.FLAG_TIME);
                        }

                        taskUpdate = {
                            timeSpent: totalTimeSpent,
                            endTime: now,
                            status: "completed"	//NO I18N
                        };

                        let responseText = `*Task Name:* ${taskDetails.NAME}\n*Project:* ${taskDetails.PROJECT_TITLE}\n\n`;
                        responseText += `*Started at:* ${CommonUtil.getLocaleStringFromMS(taskDetails.START_TIME, "datetime")}\n`;  //NO I18N
                        responseText += `*Completed at:* ${CommonUtil.getLocaleStringFromMS(now, "datetime")}\n`;   //NO I18N
                        responseText += `*Time spent*: ${CommonUtil.getDurationFromMS(totalTimeSpent)}`;

                        response = {
                            text: responseText,
                            card: {
                                title: ":task: Task Completed",	//NO I18N
                                theme: "modern-inline"	//NO I18N
                            }
                        };
                        await CommonUtil.postMessageToBot(
                            await DatabaseUtil.users.getToken(zuid),
                            response
                        );
                        response = {};
                    }
                }
                await DatabaseUtil.tasks.updateTask(taskId, taskUpdate);
            }
        }
        return response;
    };

    const _deleteProject = async (projectId) => {
        await DatabaseUtil.projects.deleteProject(projectId);
        return CommonUtil.getBannerResponse("Project deleted.");	//NO I18N
    }

    return {
        handler: _handler
    };
})();

module.exports = Functions;